// Decompressor
// Josh Wade
// 11/6/2018
// Part two of a two part assignment
// Unpacks the data from the bin file and assigns it a compressed value
// Prints out the data to a .txt file as compressed floats
// Also calculates the RMS values to find out how off the compression was

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>
#include <string.h>

// Prototypes
int readArguments(int argc, char *argv[]);
int readBucketData();
int createFileNames();
int openInputFiles();
int openOutputFiles();
int buildBuckets();
float getBucketValue(unsigned long number);
int decompressFromBits();
unsigned short parseFromBuffer(unsigned short readableBits);
void appendToRMSSum(float val1, float val2);
float calculateRMS(int totalElements);

// Sizes for different data types
char charLength = 8;
char shortLength = 16;
char longLength = 32;

// File names and their components
char fullOutputFileName[30];
char fullBinFileName[30];
char fullVertFileName[30];
char fullRMSFileName[30];
char binaryFileName[] = "binary";
char vertsCleanedFileName[] = "vertsCleaned";
char vertsCompressedFileName[] = "vertsCompressed";
char rmsFileName[] = "RMS";
char binaryExtension[] = ".bin";
char textExtension[] = ".txt";

// File pointers 
FILE *inputBinFile = NULL;
FILE *inputVertFile = NULL;
FILE *outputVertFile = NULL;
FILE *outputRMSFile = NULL;

// How many bits are in our compressed data points
unsigned short bitCount;

// Min and max values of the vertices
float maxValue = FLT_MIN;
float minValue = FLT_MAX;

// How big each "bucket" of data is
float bucketSize = 0;

// How many "buckets" there are 
unsigned long bucketCount = 0;

// How many elements were compressed
int compressedElements = 0;

// Our temporary store for the short we read in 
unsigned long parseBuffer = 0;

// A going sum to track the totals for RMS calculation
float runningRMSSum = 0;

int main(int argc, char *argv[])
{
	if (readArguments(argc, argv))
		return 1;

	if (createFileNames())
		return 1;

	if (openInputFiles())
		return 1;

	if (openOutputFiles())
		return 1;

	if (readBucketData())
		return 1;

	if (buildBuckets())
		return 1;

	if (decompressFromBits())
		return 1;

	// Close the files 
	fclose(inputBinFile);
	fclose(inputVertFile);
	fclose(outputVertFile);

	return 0;
}

// Read in command line arguments
int readArguments(int argc, char *argv[])
{
	// Check if the argument count is correct
	if (argc != 2) {
		printf("Invalid argument list! Please try again.");
		getchar();
		return 1;
	}

	// Read in file name & bit count
	bitCount = (unsigned short)atoi(argv[1]);

	bucketCount = (unsigned long)(pow(2, bitCount));

	return 0;
}

// Reads initial metadata for the bucket information
int readBucketData()
{
	char line[256];

	fgets(line, sizeof(line), inputBinFile);
	sscanf(line, "%f %f %d", &minValue, &maxValue, &compressedElements);

	// Since every vert is composed of 3 elements, we're actually looking at 3 times that many elements
	compressedElements = compressedElements * 3;

	return 0;
}

// Creates the file names based on our bit counts
int createFileNames()
{
	// Parse our bit count as a string
	char bitCountAsString[3];
	_itoa(bitCount, bitCountAsString, 10);

	// Build the bin file name with the bit count in it
	strcat(fullBinFileName, binaryFileName);
	strcat(fullBinFileName, bitCountAsString);
	strcat(fullBinFileName, binaryExtension);

	// Build the cleaned vert file name
	// This originally contained the bitCount, but this has been considered redundant
	strcat(fullVertFileName, vertsCleanedFileName);
	strcat(fullVertFileName, textExtension);

	// Build the RMS output file name
	strcat(fullRMSFileName, rmsFileName);
	strcat(fullRMSFileName, bitCountAsString);
	strcat(fullRMSFileName, textExtension);

	// Build the compressed vert file name with bit count in it 
	strcat(fullOutputFileName, vertsCompressedFileName);
	strcat(fullOutputFileName, bitCountAsString);
	strcat(fullOutputFileName, textExtension);

	return 0;
}

// Opens the input files
int openInputFiles()
{
	// Check if each file correctly opened
	inputBinFile = fopen(fullBinFileName, "rb");
	inputVertFile = fopen(fullVertFileName, "r");
	if (inputBinFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullBinFileName);
		getchar();
		return 1;
	}
	if (inputVertFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullVertFileName);
		getchar();
		return 1;
	}
	return 0;
}

// Opens the output files
int openOutputFiles()
{
	// Check if the files correctly opened
	outputVertFile = fopen(fullOutputFileName, "w");
	if (outputVertFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullOutputFileName);
		getchar();
		return 1;
	}
	outputRMSFile = fopen(fullRMSFileName, "w");
	if (outputRMSFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullRMSFileName);
		getchar();
		return 1;
	}
	return 0;
}

// Figure out the sizes of our buckets 
int buildBuckets()
{
	// Warn if we're attempting a by-0 division 
	if (bucketCount < 1) {
		return 1;
	}

	// Determine the increment size for one bucket to the next 
	bucketSize = (float(maxValue - minValue)) / (bucketCount);
	return 0;
}

// Figures out the compressed value for this data point
float getBucketValue(unsigned long number)
{
	// The compressed value is equal to the average of that bucket's boundaries
	return ((minValue + (number * bucketSize)) +
		(minValue + ((number + 1) * bucketSize))) / 2.0f;
}

// Unpacks the data from the stream
int decompressFromBits()
{
	// Init value and spaceOpen for shifting and reading purposes
	unsigned short value, spaceOpen = 0;
	
	// Assign a temp value to track our loop
	int totalElements = compressedElements;

	while (compressedElements > 0)
	{
		// Read the data and move the bits to our long parseBuffer
		// Since our data is a bitstream of shorts, a long buffer allows intuitive pipelining
		// we can read the top few bits while shifting in the next short as soon as there are not enough bits to read
		fread(&value, sizeof(short), 1, inputBinFile);
		unsigned long movedValue = value;
		movedValue = movedValue << (shortLength - spaceOpen);
		parseBuffer |= movedValue;

		// Once there is a guarantee of readable bits at the top of the parse buffer, parse it
		spaceOpen = parseFromBuffer(shortLength + spaceOpen);
	}

	// Calculate the RMS
	fprintf(outputRMSFile, "%.5f\n", calculateRMS(totalElements));
	return 0;
}

// Reads out data from the buffer
unsigned short parseFromBuffer(unsigned short readableBits)
{
	// Continue as long as there are bits we can read from the buffer
	while (readableBits >= bitCount)
	{
		// Mask the top bitCount bits, then adjust the parse buffer
		unsigned long bitsToRead = parseBuffer;
		parseBuffer = parseBuffer << bitCount;
		bitsToRead = (bitsToRead >> (longLength - bitCount));

		// Acquire the compressed bucket value
		float bucketToPrint = getBucketValue(bitsToRead);

		// Print out the compressed value
		fprintf(outputVertFile, "%.5f\n", bucketToPrint);

		// Compare to the uncompressed value and add to the rms sum
		float uncompressedRead = 0;
		fscanf(inputVertFile, "%f", &uncompressedRead);
		appendToRMSSum(bucketToPrint, uncompressedRead);

		// Check if we've read every data point
		compressedElements--;
		if (compressedElements == 0)
			return 0;

		// Reduce our readable bits
		readableBits -= bitCount;
	}

	// Give back however many bits are readable for shifting purposes
	return readableBits;
}

// Substract one value from the other, squaring the result, then add to running sum
void appendToRMSSum(float val1, float val2)
{
	runningRMSSum += (val1 - val2) * (val1 - val2);
}

// Divide by number of elements and square root the value 
float calculateRMS(int totalElements)
{
	return sqrt(runningRMSSum / totalElements);
}